#!/bin/sh

# This software originates from Freifunk Berlin and registers nodes
# from the Freifunk Berlin Network at our online map at https://openwifimap.net.
# This is a reimplementation of a former lua-script.
# It is licensed under GNU General Public License v3.0 or later
# Copyright (C) 2021   Patrick Grimm
# Copyright (C) 2021   Martin Hübner

# Omit warning for missing local statement. busybox-ash has them included
# shellcheck shell=dash

# jshn assigns the variables for us, but shellcheck doesn't get this.
# shellcheck disable=SC2154
# We use printf for consistency, even though it has no vars.
# shellcheck disable=SC2182
# We've used a hack for embedding a json-string into a json-string with sed. Having '$OLSRCONFIG' in single-quotes is by intention.
# shellcheck disable=SC2016
# using printf with variables and nc didn't work correctly. Thus this hack
# shellcheck disable=SC2059
# by full intention: jshn needs to get numeric values without double-quotes!
# shellcheck disable=SC2086
# The unused variables remain for future use. We intend to use them later on.
# shellcheck disable=SC2034

# we can't acess those libraries at compile-time. Thus ignoring.
# shellcheck source=/dev/null
. /lib/functions.sh
. /usr/share/libubox/jshn.sh

OWM_API_VER="1.0"

printhelp() {
    printf "owm.sh - Tool for registering routers at openwifimap.net\n
Options:
\t--help|-h:\tprint this text

\t--dry-run:\tcheck if owm.lua is working (does not paste any data).
\t\t\tWith this option you can check for errors in your
\t\t\tconfiguration and test the transmission of data to
\t\t\tthe map.\n\n
If invoked without any options, this tool will try to register
your node at the community-map and print the servers response.
To work correctly, this tool will need at least the geo-location
of the node (check correct execution with --dry-run).

To override the server used by this script, set freifunk.community.owm_api.
"
}

# save positional argument, as it would get overwritten otherwise.
CMD_1="$1"
if [ -n "$CMD_1" ] && [ "$CMD_1" != "--dry-run" ]; then
    [ "$CMD_1" != "-h" ] && [ "$CMD_1" != "--help" ] && printf "Unrecognized argument %s.\n\n" "$CMD_1"
    printhelp
    exit 1
fi

# calback function: This function aggregates all items of the 'contact'
# option list from /etc/config/freifunk into one single string for better
# transport
handle_contact() {
    local value="$1"

    if [ -n "$value" ]; then
        CONTACT_AGGREGATOR="$CONTACT_AGGREGATOR|$value"
    fi
}

######################
#                    #
#  Collect OWM-Data  #
#                    #
######################

olsr4_links() {
    json_select "$2"
    json_get_var localIP localIP
    json_get_var remoteIP remoteIP
    # extract the second level domain from the host and append .olsr to be compatible with bgbdisco suffix .ff
    remotehost="$(nslookup "$remoteIP" 2>/dev/null | grep name | sed -e 's/.*name = \(.*\)/\1/' | awk -F. '{print $(NF-1)".olsr"}')"
    if [ -z "$remotehost" ]; then
        remotehost="$remoteIP"
    fi
    json_get_var linkQuality linkQuality
    json_get_var olsrInterface olsrInterface
    json_get_var ifName ifName
    json_select ..
    if ! echo "$olsrInterface" | grep -q -E '.*(wg|ts)_.*'; then
        olsr4links="$olsr4links$localIP $remoteIP $remotehost $linkQuality $ifName;"
    fi
}

# This section is relevant for hopglass statistics feature (isUplink/isHotspot)
OLSRCONFIG=$(printf "/config" | nc 127.0.0.1 9090)

# collect nodes location
uci_load system
longitude="$(uci_get system @system[-1] longitude)"
latitude="$(uci_get system @system[-1] latitude)"

#
#   set another type if lat/lon is not set.
#
type="node"
if [ -z "$latitude" ] || [ -z "$longitude" ]; then
    printf "latitude/longitude is not set.\nFYI ...\n"
    type="node_no_loc"	
fi


# collect data on OLSR-links
json_load "$(printf "/links" | nc 127.0.0.1 9090 2>/dev/null)" 2>/dev/null
#json_get_var timeSinceStartup timeSinceStartup
olsr4links=""
if json_is_a links array; then
    json_for_each_item olsr4_links links
fi
json_cleanup

# collect board info
json_load "$(ubus call system board)"
json_get_var model model
json_get_var hostname hostname
json_get_var system system
json_select release
json_get_var revision revision
json_get_var distribution distribution
json_get_var version version
json_get_var description description
json_select ..
json_load "$(ubus call system info)"
json_get_var uptime uptime
json_get_values loads load

# if file freifunk_release is available, override version and revision
if [ -f /etc/freifunk_release ]; then
    . /etc/freifunk_release
    distribution="$FREIFUNK_DISTRIB_ID"
    version="$FREIFUNK_RELEASE"
    revision="$FREIFUNK_REVISION"
fi

if [ -f /etc/weimarnetz_release ]; then
    . /etc/weimarnetz_release
    pkgdesc="$WEIMARNETZ_PACKAGES_DESCRIPTION"
    pkgbranch="$WEIMARNETZ_PACKAGES_BRANCH"
    pkgrev="$WEIMARNETZ_PACKAGES_REV"
fi

# Get Sysload
sysload=$(cat /proc/loadavg)
load1=$(echo "$sysload" | cut -d' ' -f1)
load5=$(echo "$sysload" | cut -d' ' -f2)
load15=$(echo "$sysload" | cut -d' ' -f3)

# Date when the firmware was build.
kernelString=$(cat /proc/version)
buildDate=$(echo "$kernelString" | cut -d'#' -f2 | cut -c 3-)
kernelVersion=$(echo "$kernelString" | cut -d' ' -f3)

# contact information
uci_load freifunk
name="$(uci_get freifunk contact name)"
nick="$(uci_get freifunk contact nickname)"
showMail="$(uci_get ffwizard settings email2owm)"
if [ "$showMail" -eq "1" ]; then
    mail="$(uci_get freifunk contact mail)"
    phone="$(uci_get freifunk contact phone)"
else
    mail="Email hidden"
fi
homepage="$(uci_get freifunk contact homepage)" # whitespace-separated, with single quotes, if string contains whitspace
note="$(uci_get freifunk contact note)"

# aggregate contacts-list into one string
config_load freifunk
config_list_foreach contact contact handle_contact
# omit the first pipe-symbol.
contacts=$(echo "$CONTACT_AGGREGATOR" | sed 's/|//')

# community info
ssid="$(uci_get freifunk community ssid)"
mesh_network="$(uci_get freifunk community mesh_network)"
uci_owm_api="$(uci_get freifunk community owm_api)"
owm_api_host=$(echo "$uci_owm_api" | sed -e 's/http\(s\)\{0,1\}:\/\///g')
com_name="$(uci_get freifunk community name)"
com_homepage="$(uci_get freifunk community homepage)"
com_longitude="$(uci_get freifunk community longitude)"
com_latitude="$(uci_get freifunk community latitude)"
com_ssid_scheme="$(uci_get freifunk community ssid_scheme)"
com_splash_network="$(uci_get freifunk community splash_network)"
com_splash_prefix="$(uci_get freifunk community splash_prefix)"
uci_load ffwizard
nodenumber="$(uci_get ffwizard settings nodenumber)"
json_load "$(ubus call registrator status)"
json_get_var regmessage message
json_get_var regnodenumber nodenumber
json_get_var regcode code
json_get_var regsuccess success

###########################
#                         #
#  Construct JSON-string  #
#                         #
###########################

json_init
json_add_object freifunk
{
    json_add_object contact
    {
        if [ -n "$name" ]; then json_add_string name "$name"; fi
        # contact list superseeds the use of mail option
        if [ -n "$contacts" ]; then
            json_add_string mail "$contacts"
        else
            if [ -n "$mail" ]; then json_add_string mail "$mail"; fi
        fi
        if [ -n "$nick" ]; then json_add_string nickname "$nick"; fi
        if [ -n "$phone" ]; then json_add_string phone "$phone"; fi
        if [ -n "$homepage" ]; then json_add_string homepage "$homepage"; fi # was array of homepages
        if [ -n "$note" ]; then json_add_string note "$note"; fi
    }
    json_close_object

    json_add_object community
    {
        json_add_string ssid "$ssid"
        json_add_string mesh_network "$mesh_network"
        json_add_string owm_api "$uci_owm_api"
        json_add_string name "$com_name"
        json_add_string homepage "$com_homepage"
        json_add_string longitude "$com_longitude"
        json_add_string latitude "$com_latitude"
        json_add_string ssid_scheme "$com_ssid_scheme"
        json_add_string splash_network "$com_splash_network"
        json_add_int splash_prefix $com_splash_prefix
    }
    json_close_object
}
json_close_object

# script infos
json_add_string type "$type"
json_add_string script "owm.sh"
json_add_double api_rev $OWM_API_VER

json_add_object system
{
    json_add_array sysinfo
    {
        json_add_string "" "system is deprecated"
        json_add_string "" "$model"
    }
    json_close_array
    json_add_array uptime
    {
        json_add_int "" $uptime
    }
    json_close_array
    json_add_array loadavg
    {
        json_add_double "" $load5
    }
    json_close_array
}
json_close_object

# OLSR-Config
# That string gets substituted by the olsrd-config-string afterwards
json_add_object olsr
send_olsrd_config="$(uci_get ffwizard owm send_olsrd_config '1')"
if [ "$send_olsrd_config" = "1" ]; then
    json_add_string ipv4Config '$OLSRCONFIG'
fi
json_close_object

json_add_array links
{
    IFSORIG="$IFS"
    IFS=';'
    for i in ${olsr4links}; do
        IFS="$IFSORIG"
        set -- $i
        json_add_object
        {
            json_add_string sourceAddr4 "$1"
            json_add_string destAddr4 "$2"
            json_add_string id "$3"
            json_add_double quality "$4"
        }
        json_close_object
        IFS=';'
    done
    IFS="$IFSORIG"
}
json_close_array


# General node info
# Bug in add_double function. Mostly it adds unwanted digits
# but they disappear, if we send stuff to the server
if [ -n "$latitude" ] && [ -n "$longitude" ]; then
	json_add_double latitude $latitude
	json_add_double longitude $longitude
fi
json_add_string hostname "$hostname"
json_add_int updateInterval 1800
json_add_string hardware "$system"
json_add_object firmware
{
    json_add_string name "$distribution $version"
    json_add_string distname "$distribution $version"
    json_add_string distversion "$version"
    json_add_string revision "$revision"
    json_add_string description "$description"
    json_add_string kernelVersion "$kernelVersion"
    json_add_string kernelBuildDate "$buildDate"
    json_add_string packageDescription "$pkgdesc"
    json_add_string packageBranch "$pkgbranch"
    json_add_string packageRevision "$pkgrev"
}
json_close_object
json_add_object weimarnetz
{
    json_add_string "nodenumber" "$nodenumber"
    json_add_object registratorstatus
    {
        json_add_string message "$regmessage"
        json_add_int nodenumber $regnodenumber
        json_add_boolean success $regsuccess
        json_add_int code "$regcode"
    }
    json_close_object
}
json_close_object

JSON_STRING=$(json_dump)
# insert json-string from OLSR and repair wrong syntax at string-borders (shell-quotes...)
JSON_STRING=$(echo "$JSON_STRING" | sed -e 's|$OLSRCONFIG|'"$OLSRCONFIG"'|; s|"{|{|; s|}"|}|')

# just print data to stdout, if we have test-run.
if [ "$CMD_1" = "--dry-run" ]; then
	printf "%s\n" "$JSON_STRING"
	exit 0
fi


################################
#                              #
#   Send data to openwifimap   #
#                              #
################################

#echo $JSON_STRING
# get message lenght for request
LEN=${#JSON_STRING} 

MSG="\
PUT /update_node/$hostname.olsr HTTP/1.1\r
User-Agent: nc/0.0.1\r
Host: $owm_api_host\r
Content-type: application/json\r
Content-length: $LEN\r
\r
$JSON_STRING\r\n"

printf "$MSG" | nc "$owm_api_host" 80
printf "\n\n"
